<?php
// view_details.php
include '../db.php'; // Include the database connection file

// Check if an application ID is passed in the URL
if (!isset($_GET['id']) || empty($_GET['id'])) {
    echo "Invalid request: No application ID provided.";
    exit;
}

$app_id = $_GET['id'];

// Handle file upload logic
function handle_upload($file_key, $current_filename) {
    $posted_filename = $_POST[$file_key . '_hidden'] ?? $current_filename;

    // If file uploaded
    if (isset($_FILES[$file_key]) && $_FILES[$file_key]['error'] == UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/';
        $new_filename = time() . '_' . basename($_FILES[$file_key]['name']);
        $target_path = $upload_dir . $new_filename;
        if (move_uploaded_file($_FILES[$file_key]['tmp_name'], $target_path)) {
            if (!empty($posted_filename) && file_exists($upload_dir . $posted_filename)) {
                unlink($upload_dir . $posted_filename);
            }
            return $new_filename;
        }
    }

    // If hidden input is empty, delete current file
    if (empty($posted_filename) && !empty($current_filename) && file_exists('../uploads/' . $current_filename)) {
        unlink('../uploads/' . $current_filename);
    }

    return $posted_filename;
}




// Fetch the application details from the database
$sql = "SELECT * FROM applications WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $app_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows > 0) {
    $application = $result->fetch_assoc();
} else {
    echo "No application found with ID: " . htmlspecialchars($app_id);
    exit;
}

// Handle form submission for updating application details
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // --- Handle File Uploads ---
    $passport_photo = handle_upload('passport_photo', $application['passport_photo']);
    $license_front  = handle_upload('license_front', $application['license_front']);
    $license_back   = handle_upload('license_back', $application['license_back']);
    $license_class  = handle_upload('license_class', $application['license_class']);

    // --- Collect all fields explicitly except 'status' ---
    $fields = [
        'full_name', 'dob', 'gender', 'email', 'phone', 'nationality',
        'license_number', 'plan', 'type', 'license_country', 'drive_country',
        'addons', 'promo', 'ship_fname', 'ship_lname', 'street', 'state',
        'zip', 'city', 'country', 'product', 'validity'
    ];

    foreach ($fields as $field) {
        $$field = $_POST[$field] ?? $application[$field]; // Create variable dynamically
    }

    // --- Collect status safely ---
    $status_options = ['active','inactive','completed','in-progress','shipped','cancelled','pending']; 
    $status = strtolower($_POST['status'] ?? $application['status']); // convert to lowercase
    if (!in_array($status, $status_options)) {
        $status = $application['status']; // fallback if invalid
    }

    // --- Prepare update ---
    $update_sql = "UPDATE applications SET
        full_name=?, dob=?, gender=?, email=?, phone=?, nationality=?,
        license_number=?, plan=?, type=?, license_country=?, drive_country=?,
        addons=?, promo=?, ship_fname=?, ship_lname=?, street=?, state=?,
        zip=?, city=?, country=?, product=?, validity=?, status=?,
        passport_photo=?, license_front=?, license_back=?, license_class=?
        WHERE id=?";

    $stmt = $conn->prepare($update_sql);
    $stmt->bind_param(
        'sssssssssssssssssssssssssssi',
        $full_name, $dob, $gender, $email, $phone, $nationality,
        $license_number, $plan, $type, $license_country, $drive_country,
        $addons, $promo, $ship_fname, $ship_lname, $street, $state,
        $zip, $city, $country, $product, $validity, $status,
        $passport_photo, $license_front, $license_back, $license_class,
        $app_id
    );

    if ($stmt->execute()) {
        $success_message = "Application updated successfully.";
        // Refresh application data
        $stmt = $conn->prepare("SELECT * FROM applications WHERE id = ?");
        $stmt->bind_param("i", $app_id);
        $stmt->execute();
        $application = $stmt->get_result()->fetch_assoc();
    } else {
        $error_message = "Error updating application: " . $stmt->error;
    }
}


?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Itca Dashboard - Edit Application</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link rel="stylesheet" href="assets/css/style.css" />
    <style>
        .form-section { background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: .375rem; padding: 1.5rem; margin-bottom: 2rem; }
        .form-section h4 { border-bottom: 2px solid #ccc; padding-bottom: 10px; margin-bottom: 20px; }
        .current-image { max-width: 150px; border: 1px solid #ddd; padding: 5px; margin-top: 10px; border-radius: 5px; }
    </style>
</head>
<body>
<div class="dashboard_wrapper">
    <div class="left_panel">
        <div class="logo_box">
            <img src="../assets/images/logo.png" alt="logo" class="img-fluid full_logo" />
            <img src="../assets/images/logo-icon.png" alt="logo" class="img-fluid logo_icon" />
        </div>
        <div class="menu_link">
            <ul>
                <li><a href="index.php" class="active">Permit Request</a></li>
                <li><a href="#">Form Controller</a></li>
                <li><a href="#">Profile Setting</a></li>
                <li><a href="#">Log Out</a></li>
            </ul>
        </div>
    </div>

    <div class="right_body_panel">
        <div class="right_header_panel">
            <div class="left_toggle_button">
                <button class="btn toggle_btn" type="button"></button>
            </div>
            <div class="right_user_box">
                <div class="user_box_warp">
                    <span class="circle-name">Ad</span>
                    <div class="user_name"><h4>Admin</h4></div>
                </div>
            </div>
        </div>

        <div class="body_wrapper_section">
            <div class="body_wrap">
                <div class="container-fluid mt-4">
                    <h3>Edit Application #<?= htmlspecialchars($application['id']) ?></h3>
                    <hr>
                    <?php if (isset($error_message)): ?>
                        <div class="alert alert-danger"><?= $error_message ?></div>
                    <?php endif; ?>

                    <form method="post" enctype="multipart/form-data">
                        <!-- Applicant Details Section -->
                        <div class="form-section">
                            <h4>Applicant's Details</h4>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="full_name" class="form-label">Full Name</label>
                                    <input type="text" class="form-control" id="full_name" name="full_name" value="<?= htmlspecialchars($application['full_name']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="dob" class="form-label">Date of Birth</label>
                                    <input type="date" class="form-control" id="dob" name="dob" value="<?= htmlspecialchars($application['dob']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="gender" class="form-label">Gender</label>
                                    <input type="text" class="form-control" id="gender" name="gender" value="<?= htmlspecialchars($application['gender']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($application['email']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="phone" class="form-label">Phone</label>
                                    <input type="text" class="form-control" id="phone" name="phone" value="<?= htmlspecialchars($application['phone']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="nationality" class="form-label">Nationality</label>
                                    <input type="text" class="form-control" id="nationality" name="nationality" value="<?= htmlspecialchars($application['nationality']) ?>">
                                </div>
                            </div>
                        </div>

                        <!-- Plan and License Details Section -->
                        <div class="form-section">
                            <h4>Plan and License Details</h4>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="license_number" class="form-label">License Number</label>
                                    <input type="text" class="form-control" id="license_number" name="license_number" value="<?= htmlspecialchars($application['license_number']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="plan" class="form-label">Plan</label>
                                    <input type="text" class="form-control" id="plan" name="plan" value="<?= htmlspecialchars($application['plan']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="type" class="form-label">Type</label>
                                    <input type="text" class="form-control" id="type" name="type" value="<?= htmlspecialchars($application['type']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="license_country" class="form-label">License Country</label>
                                    <input type="text" class="form-control" id="license_country" name="license_country" value="<?= htmlspecialchars($application['license_country']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="drive_country" class="form-label">Drive Country</label>
                                    <input type="text" class="form-control" id="drive_country" name="drive_country" value="<?= htmlspecialchars($application['drive_country']) ?>">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="license_class" class="form-label">License Class Image</label>
                                    <input type="file" class="form-control" id="license_class" name="license_class">
                                    <img src="../uploads/<?= htmlspecialchars($application['license_class']) ?>" alt="License Class Image" class="current-image mt-2">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="addons" class="form-label">Addons</label>
                                    <input type="text" class="form-control" id="addons" name="addons" value="<?= htmlspecialchars($application['addons']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="promo" class="form-label">Promo Code</label>
                                    <input type="text" class="form-control" id="promo" name="promo" value="<?= htmlspecialchars($application['promo']) ?>">
                                </div>
                            </div>
                        </div>

                        <!-- Documents Section -->
                        <div class="form-section">
                            <h4>Documents</h4>
                            <div class="row">
                                <div class="row">
    <!-- Passport Photo -->
    <div class="col-md-4 mb-3">
        <label for="passport_photo" class="form-label">Passport Photo</label>
        <input type="file" class="form-control" id="passport_photo" name="passport_photo">
        <?php if (!empty($application['passport_photo']) && file_exists("../uploads/{$application['passport_photo']}")): ?>
            <div class="image-wrapper mt-2">
                <img src="../uploads/<?= htmlspecialchars($application['passport_photo']) ?>" alt="Passport Photo" class="current-image" width="100">
                <button type="button"
                        class="btn btn-sm btn-danger mt-1 remove-image-btn"
                        data-image-type="passport_photo"
                        data-app-id="<?= $application['id'] ?>">
                    Remove
                </button>
            </div>
        <?php endif; ?>
    </div>

    <!-- License Front -->
    <div class="col-md-4 mb-3">
        <label for="license_front" class="form-label">License Front</label>
        <input type="file" class="form-control" id="license_front" name="license_front">
        <?php if (!empty($application['license_front']) && file_exists("../uploads/{$application['license_front']}")): ?>
            <div class="image-wrapper mt-2">
                <img src="../uploads/<?= htmlspecialchars($application['license_front']) ?>" alt="License Front" class="current-image" width="100">
                <button type="button"
                        class="btn btn-sm btn-danger mt-1 remove-image-btn"
                        data-image-type="license_front"
                        data-app-id="<?= $application['id'] ?>">
                    Remove
                </button>
            </div>
        <?php endif; ?>
    </div>

    <!-- License Back -->
    <div class="col-md-4 mb-3">
        <label for="license_back" class="form-label">License Back</label>
        <input type="file" class="form-control" id="license_back" name="license_back">
        <?php if (!empty($application['license_back']) && file_exists("../uploads/{$application['license_back']}")): ?>
            <div class="image-wrapper mt-2">
                <img src="../uploads/<?= htmlspecialchars($application['license_back']) ?>" alt="License Back" class="current-image" width="100">
                <button type="button"
                        class="btn btn-sm btn-danger mt-1 remove-image-btn"
                        data-image-type="license_back"
                        data-app-id="<?= $application['id'] ?>">
                    Remove
                </button>
            </div>
        <?php endif; ?>
    </div>
</div>

                        </div>

                        <!-- Shipping Address Section -->
                        <div class="form-section">
                            <h4>Shipping Address</h4>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="ship_fname" class="form-label">Shipping First Name</label>
                                    <input type="text" class="form-control" id="ship_fname" name="ship_fname" value="<?= htmlspecialchars($application['ship_fname']) ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="ship_lname" class="form-label">Shipping Last Name</label>
                                    <input type="text" class="form-control" id="ship_lname" name="ship_lname" value="<?= htmlspecialchars($application['ship_lname']) ?>">
                                </div>
                                <div class="col-md-12 mb-3">
                                    <label for="street" class="form-label">Street Address</label>
                                    <input type="text" class="form-control" id="street" name="street" value="<?= htmlspecialchars($application['street']) ?>">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="city" class="form-label">City</label>
                                    <input type="text" class="form-control" id="city" name="city" value="<?= htmlspecialchars($application['city']) ?>">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="state" class="form-label">State</label>
                                    <input type="text" class="form-control" id="state" name="state" value="<?= htmlspecialchars($application['state']) ?>">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="zip" class="form-label">ZIP Code</label>
                                    <input type="text" class="form-control" id="zip" name="zip" value="<?= htmlspecialchars($application['zip']) ?>">
                                </div>
                                <div class="col-md-12 mb-3">
                                    <label for="country" class="form-label">Country</label>
                                    <input type="text" class="form-control" id="country" name="country" value="<?= htmlspecialchars($application['country']) ?>">
                                </div>
                            </div>
                        </div>

                        <!-- Order Information Section -->
                        <div class="form-section">
                            <h4>Order Information</h4>
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label for="product" class="form-label">Product</label>
                                    <input type="text" class="form-control" id="product" name="product" value="<?= htmlspecialchars($application['product']) ?>">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="validity" class="form-label">Validity</label>
                                    <input type="text" class="form-control" id="validity" name="validity" value="<?= htmlspecialchars($application['validity']) ?>">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="status" class="form-label">Status</label>
                                        <select class="form-select" id="status" name="status" required>
                                            <option value="pending" <?= strtolower($application['status'])=='pending'?'selected':'' ?>>Pending</option>
                                            <option value="in-progress" <?= strtolower($application['status'])=='in-progress'?'selected':'' ?>>In Progress</option>
                                            <option value="shipped" <?= strtolower($application['status'])=='shipped'?'selected':'' ?>>Shipped</option>
                                            <option value="completed" <?= strtolower($application['status'])=='completed'?'selected':'' ?>>Completed</option>
                                            <option value="cancelled" <?= strtolower($application['status'])=='cancelled'?'selected':'' ?>>Cancelled</option>
                                        </select>
                                </div>
                            </div>
                        </div>

                        <button type="submit" class="btn btn-primary mt-3 mb-5">Save Changes</button>
                        <a href="index.php" class="btn btn-secondary mt-3 mb-5">Cancel</a>
                    </form>

                </div>
            </div>
        </div>
    </div>
</div>
</body>
<!-- jQuery and AJAX for Remove buttons -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
    $(document).on('click', '.remove-image-btn', function() {
        let imageType = $(this).data('image-type');
        let appId = $(this).data('app-id');
        let button = $(this);

        if (!confirm('Are you sure you want to remove this image?')) return;

        $.post('ajax_remove_image.php', { image: imageType, id: appId }, function(response) {
            if (response === 'success') {
                button.closest('.image-wrapper').remove();
                // Mark field as empty so form submission doesn't overwrite others
                $('#' + imageType + '_hidden').val('');
            } else {
                alert('Failed to delete image: ' + response);
            }
        });
    });
});

</script>

</html>
