<?php
// Ensure strict error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// --- Robust Inclusion and Connection Checks ---

// Include session handler
if (!include 'session.php') {
    error_log("FATAL ERROR: Failed to include session.php. Please check file existence and permissions.");
    // In a production environment, you might redirect to a custom error page.
    // For immediate debugging, dying here is also an option if errors are suppressed.
    die("Critical Error: Session handler unavailable."); 
}

// Include database connection
if (!include 'db.php') {
    error_log("FATAL ERROR: Failed to include db.php. Please check file existence and permissions.");
    die("Critical Error: Database connection unavailable.");
}

// Check if database connection is valid
if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) {
    $connectionErrorMessage = "Database connection failed or '$conn' is not a valid mysqli object.";
    if (isset($conn) && $conn instanceof mysqli) {
        $connectionErrorMessage .= " Connection error: " . $conn->connect_error;
    } else {
        $connectionErrorMessage .= " $conn is not a mysqli object.";
    }
    error_log("FATAL ERROR: " . $connectionErrorMessage);
    die("Critical Error: Database connection failed.");
}
// --- End Connection Checks ---


// Check if user has completed previous steps
if (!isset($_SESSION['step2'])) {
    header("Location: step2.php");
    exit;
}

$step1_data = $_SESSION['step1'] ?? []; // Data from step 1
$step4_saved_data = $_SESSION['step4'] ?? []; // Previously saved shipping data

// Ensure totalAmount (product price) is available
if (!isset($_SESSION['totalAmount'])) {
    // It's expected that step2 sets this. If not, redirect back to step2.
    header("Location: step2.php");
    exit;
}
$productPrice = floatval($_SESSION['totalAmount']);


$shippingOptions = []; // Initialize always to an empty array
$initialShippingOptionId = '';
$initialShippingOptionText = '';
$initialShippingPrice = 0.0;
$error_message = ''; // General error message for the user

// Handle any error messages stored in session from previous attempts (e.g., failed submission)
if (isset($_SESSION['shipping_error'])) {
    $error_message = $_SESSION['shipping_error'];
    unset($_SESSION['shipping_error']); // Clear it after retrieving
}

try {
    // Prepare and execute query to fetch available shipping rates
    $stmt = $conn->prepare("SELECT id, `option` AS option_text, price, country_code FROM shipping_rate");
    if ($stmt === false) {
        // Critical error if prepare statement fails
        throw new mysqli_sql_exception("Prepare failed for shipping rates: " . $conn->error, $conn->error_list);
    }
    if (!$stmt->execute()) {
        // Critical error if execution fails
        throw new mysqli_sql_exception("Execution failed for shipping rates: " . $stmt->error, $stmt->error_list);
    }
    $result = $stmt->get_result();

    if ($result === false) {
        // Critical error if getting result set fails
        throw new mysqli_sql_exception("Failed to get result set for shipping rates: " . $stmt->error, $stmt->error_list);
    }

    // Fetch all shipping options
    while ($row = $result->fetch_assoc()) {
        // Clean up price to ensure it's a float, handling potential non-numeric characters
        $row['price'] = floatval(preg_replace('/[^0-9.]/', '', $row['price'] ?? '0'));
        $shippingOptions[] = $row;
    }
    $stmt->close(); // Close statement

} catch (mysqli_sql_exception $e) {
    // Log the database error for server administrators
    error_log("Database error fetching shipping rates: " . $e->getMessage());
    // Set a user-friendly error message for critical shipping unavailability
    $error_message = "Sorry, shipping options are currently unavailable due to a system error. Please try again later.";
    // Ensure $shippingOptions is empty if an error occurred
    $shippingOptions = []; 
} catch (Exception $e) { // Catch any other unexpected errors during this process
    error_log("Unexpected error fetching shipping rates: " . $e->getMessage());
    $error_message = "An unexpected error occurred while fetching shipping options. Please try again.";
    $shippingOptions = [];
}


// --- Pre-selection Logic for Default Shipping Option ---

// Define mappings for specific country codes to option text parts
$shippingMap = [
    'IN' => 'To India ONLY Shipping(Express)',
    'GB' => 'Express DHL (To UK ONLY) 2-4 Working Days',
    'MY' => 'To Malaysia ONLY'
];
// List of EU countries for regional shipping
$euCountries = ['FR','DE','IT','ES','NL','BE','SE'];

// Determine the country code to use for pre-selection (from saved data or step1)
$displayCountryCode = $_SESSION['step4']['country_code'] ?? $_SESSION['step1']['country_code'] ?? '';

// Initialize variables for the default shipping option
$initialShippingOptionId = '';
$initialShippingOptionText = '';
$initialShippingPrice = 0.0;
$foundDefaultShipping = false; // Flag to track if a default option has been set

// Only attempt pre-selection if shipping options were successfully fetched
if (!empty($shippingOptions)) { 
    // 1. Try to find an exact match for the country code
    foreach ($shippingOptions as $opt) {
        if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === $displayCountryCode) {
            $initialShippingOptionId = $opt['id'];
            $initialShippingOptionText = $opt['option_text'];
            $initialShippingPrice = (float)$opt['price'];
            $foundDefaultShipping = true;
            break; // Found an exact match, stop searching
        }
    }

    // 2. If no exact match, check if the country is in the EU list
    if (!$foundDefaultShipping && !empty($euCountries) && in_array($displayCountryCode, $euCountries)) {
        foreach ($shippingOptions as $opt) {
            // Assuming 'EU' is a specific country_code value in your shipping_rate table for EU-wide shipping
            if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === 'EU') {
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = (float)$opt['price'];
                $foundDefaultShipping = true;
                break; // Found EU match, stop searching
            }
        }
    }

    // 3. If still no match, check for text matches based on $shippingMap
    if (!$foundDefaultShipping && isset($shippingMap[$displayCountryCode])) {
        $targetOptionTextPart = $shippingMap[$displayCountryCode];
        foreach ($shippingOptions as $opt) {
            if (isset($opt['option_text'], $opt['price'], $opt['id']) &&
                stripos($opt['option_text'], $targetOptionTextPart) !== false) { // Case-insensitive check
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = (float)$opt['price'];
                $foundDefaultShipping = true;
                break; // Found text match, stop searching
            }
        }
    }

    // 4. If still no match, try to find a general 'International' option
    if (!$foundDefaultShipping) {
        foreach ($shippingOptions as $opt) {
            // Assuming 'INT' is a specific country_code value for international shipping
            if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === 'INT') { 
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = (float)$opt['price'];
                break; // Found INT match, stop searching
            }
        }
    }
}

// Fallback if no shipping option was identified at all (e.g., $shippingOptions was empty or no matches found)
if ($initialShippingOptionId === '') {
    // If $error_message is already set (e.g., from DB error), keep it.
    // Otherwise, set a default "not available" message.
    if (empty($error_message)) {
        $error_message = 'Shipping options are not available for the selected destination.';
    }
    // Ensure the text and price reflect unavailability
    $initialShippingOptionText = 'Shipping not available';
    $initialShippingPrice = 0.0;
}

// Calculate the initial total price displayed on the page
$initialTotal = $productPrice + $initialShippingPrice;


// --- Handle Form Submission (POST Request) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Store submitted shipping address details in session
    $_SESSION['step4'] = [
        'first_name'      => htmlspecialchars(trim($_POST['first_name'] ?? '')),
        'last_name'       => htmlspecialchars(trim($_POST['last_name'] ?? '')),
        'ship_house'      => htmlspecialchars(trim($_POST['ship_house'] ?? '')),
        'ship_street'     => htmlspecialchars(trim($_POST['ship_street'] ?? '')),
        'city'            => htmlspecialchars(trim($_POST['city'] ?? '')), // Should be populated by JS
        'state'           => htmlspecialchars(trim($_POST['state'] ?? '')), // Should be populated by JS
        'zip'             => htmlspecialchars(trim($_POST['zip'] ?? '')),
        'country_code'    => htmlspecialchars(trim($_POST['country_code'] ?? '')), // Hidden input populated by JS
        'shipping_option' => $_POST['shipping_option'] ?? '', // ID of the selected shipping option
    ];

    $selectedShippingId = isset($_SESSION['step4']['shipping_option']) ? intval($_SESSION['step4']['shipping_option']) : 0;
    $selectedShippingRate = 0.0;
    $shippingOptionFound = false;

    // Find the price for the selected shipping option from the fetched list
    if ($selectedShippingId > 0 && !empty($shippingOptions)) {
        foreach ($shippingOptions as $option) {
            if ((int)$option['id'] === $selectedShippingId) {
                $selectedShippingRate = (float)$option['price'];
                $shippingOptionFound = true;
                break; // Found the selected option, exit loop
            }
        }
    }
    
    // Validation: If a shipping option was required but not found or invalid
    if (!$shippingOptionFound) {
        // If no valid shipping option was selected or found in the fetched list
        // Set an error message to be displayed on reload.
        if (empty($error_message)) { // If no critical DB error message already exists
            $error_message = "Please select a valid shipping method.";
        }
        $_SESSION['shipping_error'] = $error_message; // Store error in session
        header("Location: step4.php"); // Redirect back to step4 to show error
        exit;
    }

    // Store final prices in session and redirect to the next step
    $_SESSION['shipping_rate'] = $selectedShippingRate;
    $_SESSION['product_price'] = $productPrice; // Product price from step2
    $_SESSION['total_price'] = $productPrice + $selectedShippingRate; // Total price calculation

    unset($_SESSION['shipping_error']); // Clear any session error message on successful submission
    header("Location: step5.php"); // Proceed to review step
    exit;
}

?>

<!DOCTYPE html>
<html>
<head>
    <title>Step 4 - Shipping Address</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <?php include('link.php'); ?>
    <style>
        .formfield label { display: block; margin-bottom: 5px; font-weight: bold; color: #333; }
        .formfield input[type="text"],
        .formfield select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .formfield input[readonly] { background-color: #e9ecef; }
        .form-check { margin-bottom: 10px; display: flex; align-items: center; }
        .form-check-input { margin-right: 10px; }
        .form-check-label { flex-grow: 1; }
        .step-four { display: flex; gap: 10px; }
        .step-four > div { flex: 1; }
        #total-price { text-align: right; font-size: 20px; font-weight: bold; margin-top: 20px; }
        .error-message { color: red; font-weight: bold; margin-top: 10px; }
    </style>
</head>
<body>
<?php include('header.php'); ?>

<section class="step-section steplicence">
    <div class="step-header">
        <h1>Apply for International Driving License</h1>
        <p>Most accepted at the global level & trusted by travellers all around the world</p>
    </div>
    <div class="step-progress">
        <div class="step"><span class="number">01</span><span class="label">Applicant’s Details</span></div>
        <div class="step"><span class="number">02</span><span class="label">Choose Plan</span></div>
        <div class="step"><span class="number">03</span><span class="label">Add Document</span></div>
        <div class="step active"><span class="number">04</span><span class="label">Shipping</span></div>
        <div class="step"><span class="number">05</span><span class="label">Review</span></div>
        <div class="step"><span class="number">06</span><span class="label">Payment</span></div>
    </div>
</section>

<div class="licenceform">
    <div class="container">
        <div class="main-form-container">
            <div class="form-container">
                <h2>Shipping Address</h2>
                
                <?php if (!empty($error_message)): ?>
                    <p class="error-message"><?= htmlspecialchars($error_message) ?></p>
                <?php endif; ?>

                <!-- Hidden display for selected origin country code for debugging -->
                <p style="display:none">Selected Origin: <span id="selected-origin-display"><?= htmlspecialchars($displayCountryCode ?: 'N/A') ?></span></p>

                <form method="post" action="" id="shippingForm" class="form_box">
                    <div class="formfield step-four">
                        <div style="flex:1;">
                            <label>First Name *</label>
                            <input type="text" name="first_name" value="<?= htmlspecialchars($step4_saved_data['first_name'] ?? '') ?>" required>
                        </div>
                        <div style="flex:1;">
                            <label>Last Name *</label>
                            <input type="text" name="last_name" value="<?= htmlspecialchars($step4_saved_data['last_name'] ?? '') ?>" required>
                        </div>
                    </div>

                    <div class="formfield full-field">
                        <label>House / Flat / Block No *</label>
                        <input type="text" name="ship_house" value="<?= htmlspecialchars($step4_saved_data['ship_house'] ?? '') ?>" required>
                    </div>

                    <div class="formfield full-field">
                        <label>Street / Area *</label>
                        <input type="text" name="ship_street" value="<?= htmlspecialchars($step4_saved_data['ship_street'] ?? '') ?>" required>
                    </div>

                    <div class="formfield full-field"> 
                        <label>Country *</label>
                        <select id="country" name="country_code_select" required>
                            <option value="" selected>-- Select Country --</option>
                            <!-- Options will be populated by JavaScript -->
                        </select>
                        <!-- Hidden input to store the actual country code, updated by JS -->
                        <input type="hidden" id="country_hidden" name="country_code" value="<?= htmlspecialchars($displayCountryCode) ?>">
                    </div>

                    <div class="formfield full-field">
                        <label>State/Region/Province *</label>
                        <input type="text" id="state" name="state" value="<?= htmlspecialchars($step4_saved_data['state'] ?? '') ?>">
                    </div>

                    <div class="formfield step-four">
                        <div style="flex:1;">
                            <label>ZIP / Postal Code *</label>
                            <input type="text" id="zip" name="zip" value="<?= htmlspecialchars($step4_saved_data['zip'] ?? '') ?>" required>
                        </div>
                        <div style="flex:1;">
                            <label>City *</label>
                            <input type="text" id="city" name="city" value="<?= htmlspecialchars($step4_saved_data['city'] ?? '') ?>" readonly required>
                        </div>
                    </div>

                    <!-- Shipping Method Selection -->
                    <div class="formfield full-field" id="shipping-method">
                        <label>Shipping Method *</label>
                        <?php
                        // Display shipping options or error message based on server-side logic
                        if (!empty($error_message)) {
                            // If a critical error occurred during fetching shipping, display it.
                            echo "<p class='error-message'>" . htmlspecialchars($error_message) . "</p>";
                            // Provide hidden fields for form submission, reflecting no valid selection.
                            echo '<input type="hidden" name="shipping_option" value="0">';
                            echo '<input type="hidden" id="shipping_price_hidden" name="shipping_price" value="0.00">';
                        } elseif (!empty($initialShippingOptionId)) {
                            // If a default shipping option was pre-selected successfully
                            echo '<div class="form-check">';
                            echo '<input class="form-check-input" type="radio" name="shipping_option" id="shippingOption_' . htmlspecialchars($initialShippingOptionId) . '" value="' . htmlspecialchars($initialShippingOptionId) . '" checked required data-price="' . number_format($initialShippingPrice, 2, '.', '') . '">';
                            echo '<label class="form-check-label" for="shippingOption_' . htmlspecialchars($initialShippingOptionId) . '">';
                            echo htmlspecialchars($initialShippingOptionText) . ' (£' . number_format($initialShippingPrice, 2) . ')';
                            echo '</label>';
                            echo '</div>';
                            // Hidden input for the pre-selected shipping price
                            echo '<input type="hidden" id="shipping_price_hidden" name="shipping_price" value="' . number_format($initialShippingPrice, 2, '.', '') . '">';
                        } else {
                            // If no default option was found, but no critical error occurred (e.g., no options for country)
                            echo "<p>No shipping options available for the selected destination.</p>";
                            echo '<input type="hidden" name="shipping_option" value="0">';
                            echo '<input type="hidden" id="shipping_price_hidden" name="shipping_price" value="0.00">';
                        }
                        ?>
                    </div>
                    
                    <!-- Display total price -->
                    <p id="total-price">Total: £<?= number_format($initialTotal, 2) ?></p>

                    <!-- Navigation and Submit Buttons -->
                    <div class="back-continue-button-wrap">
                        <a href="step3.php"><button type="button" class="first-btn-cicle">Back</button></a>
                        
                        <!-- Submit button should be enabled if a shipping option is selected or if there's an error but user might be forced to proceed -->
                        <button type="submit" class="btn secondary-btn-circle">Continue</button>
                    </div>
                </form>
            </div>

            <div class="global-container">
                <?php include('stats-card.php'); ?>
            </div>
        </div>
    </div>
</div>

<?php include('footer.php'); ?>

<script>
document.addEventListener("DOMContentLoaded", function () {
    const countrySelect = document.getElementById("country");
    const countryHidden = document.getElementById("country_hidden");
    const zipInput = document.getElementById("zip");
    const cityInput = document.getElementById("city");
    const stateInput = document.getElementById("state");
    const shippingMethodDiv = document.getElementById("shipping-method");
    const totalPriceElem = document.getElementById("total-price");
    const selectedOriginDisplay = document.getElementById("selected-origin-display"); // For debugging
    const shippingPriceHiddenInput = document.getElementById("shipping_price_hidden");
    const shippingForm = document.getElementById("shippingForm"); 
    const submitButton = shippingForm.querySelector('button[type="submit"]');

    const productPrice = parseFloat("<?= $productPrice ?>"); // Ensure it's a float
    // Initialize currentShippingPrice from the hidden input value, which reflects the server-side pre-selection
    let currentShippingPrice = parseFloat(shippingPriceHiddenInput.value || 0); 

    // --- Helper Functions ---
    function escapeHtml(unsafe) {
        return unsafe
             .replace(/&/g, "&amp;")
             .replace(/</g, "&lt;")
             .replace(/>/g, "&gt;")
             .replace(/"/g, "&quot;")
             .replace(/'/g, "&#039;");
    }

    function updateTotalPriceDisplay() {
        const total = productPrice + currentShippingPrice;
        totalPriceElem.textContent = "Total: £" + total.toFixed(2);
    }
    
    // Function to update the submit button's enabled/disabled state
    function updateSubmitButtonState() {
        const selectedShippingRadio = document.querySelector('input[name="shipping_option"]:checked');
        const hasDbError = "<?= !empty($error_message) ?>"; // Check if a critical DB error was displayed
        
        // Generally, allow submission if a shipping option is selected,
        // OR if there was a DB error and the user might be stuck.
        // Disable if no shipping is available AND no DB error occurred AND no initial option was pre-selected.
        
        // If shipping is unavailable (price is 0 AND no valid option selected), AND no error was displayed, disable.
        if (currentShippingPrice === 0 && !selectedShippingRadio && !hasDbError && !<?= !empty($initialShippingOptionId) ? 'true' : 'false' ?>) {
             if(submitButton) submitButton.disabled = true;
        } else {
             if(submitButton) submitButton.disabled = false;
        }
    }

    // --- AJAX Functions ---

    // Fetches and displays shipping options based on country and ZIP
    function fetchAndDisplayShipping(countryCode, zipCode) {
        // Immediately clear previous results and show a loading indicator
        shippingMethodDiv.innerHTML = '<p>Loading shipping options...</p>';
        currentShippingPrice = 0.0; // Reset price
        shippingPriceHiddenInput.value = "0.00"; // Reset hidden input
        updateTotalPriceDisplay(); // Update total price display to reflect reset price

        // If no country is selected, we can't fetch shipping
        if (!countryCode) {
            shippingMethodDiv.innerHTML = "<p>Please select a country to see shipping options.</p>";
            updateSubmitButtonState(); // Update button state (might disable it if country is needed)
            return;
        }

        // Construct URL for the AJAX request
        const url = `get_shipping.php?country=${encodeURIComponent(countryCode)}&zip=${encodeURIComponent(zipCode || '')}`;

        fetch(url)
            .then(response => {
                if (!response.ok) {
                    // Handle HTTP errors (e.g., 404, 500 on get_shipping.php itself)
                    throw new Error(`HTTP error! Status: ${response.status}`);
                }
                return response.json(); // Parse JSON response
            })
            .then(data => {
                // Process the JSON data
                if (data && data.id && data.price !== undefined) { 
                    // Successfully received a shipping option
                    const priceFloat = parseFloat(data.price);
                    shippingMethodDiv.innerHTML = `
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="shipping_option" id="shippingOption_${data.id}" value="${data.id}" checked required data-price="${priceFloat.toFixed(2)}">
                            <label class="form-check-label" for="shippingOption_${data.id}">
                                ${escapeHtml(data.option)} (£${priceFloat.toFixed(2)})
                            </label>
                        </div>
                        <input type="hidden" name="shipping_price" value="${priceFloat.toFixed(2)}">
                    `;
                    currentShippingPrice = priceFloat;
                    shippingPriceHiddenInput.value = currentShippingPrice.toFixed(2); 
                } else if (data && data.error) { 
                    // The get_shipping.php script returned an error message in its JSON response
                    shippingMethodDiv.innerHTML = `<p class='error-message'>${escapeHtml(data.error)}</p>`;
                    currentShippingPrice = 0.0;
                    shippingPriceHiddenInput.value = "0.00";
                } else { 
                    // No shipping options returned, or data format was unexpected
                    shippingMethodDiv.innerHTML = "<p>No shipping options available for the selected destination.</p>";
                    currentShippingPrice = 0.0;
                    shippingPriceHiddenInput.value = "0.00";
                }
                updateTotalPriceDisplay(); // Update total price
                updateSubmitButtonState(); // Update button state based on new shipping status
            })
            .catch(error => {
                // Handle network errors or errors during JSON parsing
                console.error('Error fetching shipping:', error);
                shippingMethodDiv.innerHTML = "<p class='error-message'>Error loading shipping options. Please try again later.</p>";
                currentShippingPrice = 0.0;
                shippingPriceHiddenInput.value = "0.00";
                updateTotalPriceDisplay();
                updateSubmitButtonState(); // Update button state
            });
    }

    // Fetches and displays city/state based on ZIP code and country
    function fetchAndDisplayLocation(zipCode, countryCode) {
        // If ZIP or Country is missing, clear the fields
        if (!zipCode || !countryCode) {
            cityInput.value = '';
            stateInput.value = '';
            return;
        }

        const url = `get_location.php?zip=${encodeURIComponent(zipCode)}&country=${encodeURIComponent(countryCode)}`;

        fetch(url)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! Status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                cityInput.value = data.city || '';
                stateInput.value = data.state || '';
            })
            .catch(error => {
                console.error('Error fetching location:', error);
                cityInput.value = '';
                stateInput.value = '';
            });
    }

    // --- Event Listeners ---

    // Timer for debouncing ZIP code input
    let typingTimer;
    const typingDelay = 500; // Delay in milliseconds

    // Event listener for ZIP code input: triggers location and shipping fetches after a delay
    zipInput.addEventListener("input", function () {
        clearTimeout(typingTimer); // Clear previous timer
        const zipCode = zipInput.value.trim();
        const selectedCountry = countryHidden.value; // Get the currently selected country code

        typingTimer = setTimeout(() => {
            fetchAndDisplayLocation(zipCode, selectedCountry);
            fetchAndDisplayShipping(selectedCountry, zipCode);
        }, typingDelay);
    });

    // Fetch countries list when the DOM is ready
    fetch("get_countries.php")
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! Status: ${response.status}`);
        }
        return response.json();
    })
    .then(countries => {
        // Populate the country select dropdown
        const defaultOption = new Option("-- Select Country --", "");
        countrySelect.appendChild(defaultOption);

        countries.forEach(c => {
            const option = new Option(c.name, c.code);
            countrySelect.appendChild(option);
        });

        // Preselect the saved country if it exists and is in the list
        const savedCountryCode = countryHidden.value;
        if (savedCountryCode && countrySelect.querySelector(`option[value="${savedCountryCode}"]`)) {
            countrySelect.value = savedCountryCode;
        } else {
            countrySelect.value = ""; // Default to placeholder if saved country is invalid or not found
        }

        // Update hidden input and display text based on the selected/preselected country
        countryHidden.value = countrySelect.value;
        selectedOriginDisplay.textContent = countrySelect.options[countrySelect.selectedIndex].text || 'N/A';

        // Fetch initial location and shipping based on the selected country and any pre-filled ZIP
        fetchAndDisplayLocation(zipInput.value.trim(), countrySelect.value);
        fetchAndDisplayShipping(countrySelect.value, zipInput.value.trim());
    })
    .catch(error => {
        // Handle errors during country loading
        console.error("Error loading countries:", error);
        countrySelect.innerHTML = '<option value="">Error loading countries</option>';
        if (selectedOriginDisplay) selectedOriginDisplay.textContent = 'Error loading';
    });

    // Event listener for when the country selection changes
    countrySelect.addEventListener("change", () => {
        const selectedCountryCode = countrySelect.value;
        countryHidden.value = selectedCountryCode; // Update hidden input with selected country code
        selectedOriginDisplay.textContent = countrySelect.options[countrySelect.selectedIndex].text || 'N/A'; // Update display text

        // Clear location fields and fetch new location/shipping based on the new country
        cityInput.value = '';
        stateInput.value = '';
        fetchAndDisplayShipping(selectedCountryCode, zipInput.value.trim()); // Fetch shipping for new country
        fetchAndDisplayLocation(zipInput.value.trim(), selectedCountryCode); // Fetch location for new country
    });

    // --- Initial Setup ---
    // Set the initial total price and update button state on page load
    updateTotalPriceDisplay();
    updateSubmitButtonState();
});
</script>
</body>
</html>