<?php
session_start();

include 'session.php';
include 'db.php';

if (!isset($_SESSION['step2'])) {
    header("Location: step2.php");
    exit;
}

$step1_data = $_SESSION['step1'] ?? [];
$step4_saved_data = $_SESSION['step4'] ?? [];

if (!isset($_SESSION['totalAmount'])) {
    header("Location: step2.php");
    exit;
}
$productPrice = floatval($_SESSION['totalAmount']);

$shippingOptions = [];
$initialShippingOptionId = '';
$initialShippingOptionText = '';
$initialShippingPrice = 0.0;
$error_message = '';

if (isset($_SESSION['shipping_error'])) {
    $error_message = $_SESSION['shipping_error'];
    unset($_SESSION['shipping_error']);
}

try {
    $result = $conn->query("SELECT id, option AS option_text, price, country_code FROM shipping_rate");
    if ($result === false) {
        throw new mysqli_sql_exception("Failed to fetch shipping rates: " . $conn->error, $conn->error_list);
    }
    while ($row = $result->fetch_assoc()) {
        $row['price'] = floatval(preg_replace('/[^0-9.]/', '', $row['price']));
        $shippingOptions[] = $row;
    }
} catch (mysqli_sql_exception $e) {
    error_log("Database error fetching shipping rates: " . $e->getMessage());
}

$shippingMap = [
    'IN' => 'To India ONLY Shipping(Express)',
    'GB' => 'Express DHL (To UK ONLY) 2-4 Working Days',
    'MY' => 'To Malaysia ONLY'
];
$euCountries = ['FR','DE','IT','ES','NL','BE','SE'];
$euShipping = 'Express DHL (To EU ONLY) EUROPE';
$defaultShipping = 'International DHL';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $_SESSION['step4'] = [
        'first_name'      => htmlspecialchars(trim($_POST['first_name'] ?? '')),
        'last_name'       => htmlspecialchars(trim($_POST['last_name'] ?? '')),
        'ship_house'      => htmlspecialchars(trim($_POST['ship_house'] ?? '')),
        'ship_street'     => htmlspecialchars(trim($_POST['ship_street'] ?? '')),
        'city'            => htmlspecialchars(trim($_POST['city'] ?? '')),
        'state'           => htmlspecialchars(trim($_POST['state'] ?? '')),
        'zip'             => htmlspecialchars(trim($_POST['zip'] ?? '')),
        'country_code'    => htmlspecialchars(trim($_POST['country_code'] ?? '')),
        'shipping_option' => $_POST['shipping_option'] ?? '',
    ];

    $selectedShippingId = intval($_SESSION['step4']['shipping_option']);
    $selectedShippingRate = 0.0;

    if ($selectedShippingId > 0) {
        foreach ($shippingOptions as $option) {
            if ((int)$option['id'] === $selectedShippingId) {
                $selectedShippingRate = (float)$option['price'];
                break;
            }
        }
        if ($selectedShippingRate == 0.0 && isset($_POST['shipping_price'])) {
            $selectedShippingRate = (float) $_POST['shipping_price'];
        }
    } else {
        $error_message = "Please select a shipping method.";
        $_SESSION['shipping_error'] = $error_message;
        header("Location: step4.php");
        exit;
    }

    $_SESSION['shipping_rate'] = $selectedShippingRate;
    $_SESSION['product_price'] = $productPrice;
    $_SESSION['total_price'] = $productPrice + $selectedShippingRate;

    unset($_SESSION['shipping_error']);
    header("Location: step5.php");
    exit;
}

$displayCountryCode = $_SESSION['step4']['country_code'] ?? $_SESSION['step1']['country_code'] ?? '';

$initialShippingOptionId = '';
$initialShippingOptionText = '';
$initialShippingPrice = 0.0;
$foundByCountry = false;

if (!empty($shippingOptions) && is_array($shippingOptions)) {
    // 1. Exact country code match
    foreach ($shippingOptions as $opt) {
        if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === $displayCountryCode) {
            $initialShippingOptionId = $opt['id'];
            $initialShippingOptionText = $opt['option_text'];
            $initialShippingPrice = floatval($opt['price']);
            $foundByCountry = true;
            break;
        }
    }

    // 2. EU check
    if (!$foundByCountry && !empty($euCountries) && in_array($displayCountryCode, $euCountries)) {
        foreach ($shippingOptions as $opt) {
            if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === 'EU') {
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = floatval($opt['price']);
                $foundByCountry = true;
                break;
            }
        }
    }

    // 3. Other mapped countries
    if (!$foundByCountry && isset($shippingMap[$displayCountryCode])) {
        foreach ($shippingOptions as $opt) {
            if (isset($opt['option_text'], $opt['price'], $opt['id']) &&
                stripos($opt['option_text'], $shippingMap[$displayCountryCode]) !== false) {
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = floatval($opt['price']);
                $foundByCountry = true;
                break;
            }
        }
    }

    // 4. Default to International
    if (!$foundByCountry) {
        foreach ($shippingOptions as $opt) {
            if (isset($opt['country_code'], $opt['option_text'], $opt['price']) && $opt['country_code'] === 'INT') {
                $initialShippingOptionId = $opt['id'];
                $initialShippingOptionText = $opt['option_text'];
                $initialShippingPrice = floatval($opt['price']);
                break;
            }
        }
    }
}

// Fallback if nothing matched
if ($initialShippingOptionId === '') {
    $initialShippingOptionText = 'Shipping not available';
    $initialShippingPrice = 0.0;
}

$initialTotal = $productPrice + $initialShippingPrice;

?>

<!DOCTYPE html>
<html>
<head>
    <title>Step 4 - Shipping Address</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <?php include('link.php'); ?>
    <style>
        .formfield label { display: block; margin-bottom: 5px; font-weight: bold; color: #333; }
        .formfield input[type="text"],
        .formfield select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .formfield input[readonly] { background-color: #e9ecef; }
        .form-check { margin-bottom: 10px; display: flex; align-items: center; }
        .form-check-input { margin-right: 10px; }
        .form-check-label { flex-grow: 1; }
        .step-four { display: flex; gap: 10px; }
        .step-four > div { flex: 1; }
        #total-price { text-align: right; font-size: 20px; font-weight: bold; margin-top: 20px; }
        .error-message { color: red; font-weight: bold; margin-top: 10px; }
    </style>
</head>
<body>
<?php include('header.php'); ?>

<section class="step-section steplicence">
    <div class="step-header">
        <h1>Apply for International Driving License</h1>
        <p>Most accepted at the global level & trusted by travellers all around the world</p>
    </div>
    <div class="step-progress">
        <div class="step"><span class="number">01</span><span class="label">Applicant’s Details</span></div>
        <div class="step"><span class="number">02</span><span class="label">Choose Plan</span></div>
        <div class="step"><span class="number">03</span><span class="label">Add Document</span></div>
        <div class="step active"><span class="number">04</span><span class="label">Shipping</span></div>
        <div class="step"><span class="number">05</span><span class="label">Review</span></div>
        <div class="step"><span class="number">06</span><span class="label">Payment</span></div>
    </div>
</section>

<div class="licenceform">
    <div class="container">
        <div class="main-form-container">
            <div class="form-container">
                <h2>Shipping Address</h2>
                
                <?php if (isset($error_message)): ?>
                    <p class="error-message"><?= htmlspecialchars($error_message) ?></p>
                <?php endif; ?>

             <!-- <p>Product Price: £<?= number_format($productPrice, 2) ?></p> -->
                <p style="display:none">Selected Origin: <span id="selected-origin-display"><?= htmlspecialchars($displayCountryCode ?: 'N/A') ?></span></p>

                <form method="post" action="" id="shippingForm">
                    <div class="formfield step-four">
                        <div style="flex:1;">
                            <label>First Name *</label>
                            <input type="text" name="first_name" value="<?= htmlspecialchars($step4_saved_data['first_name'] ?? '') ?>" required>
                        </div>
                        <div style="flex:1;">
                            <label>Last Name *</label>
                            <input type="text" name="last_name" value="<?= htmlspecialchars($step4_saved_data['last_name'] ?? '') ?>" required>
                        </div>
                    </div>

                    <div class="formfield full-field">
                        <label>House / Flat / Block No *</label>
                        <input type="text" name="ship_house" value="<?= htmlspecialchars($step4_saved_data['ship_house'] ?? '') ?>" required>
                    </div>

                    <div class="formfield full-field">
                        <label>Street / Area *</label>
                        <input type="text" name="ship_street" value="<?= htmlspecialchars($step4_saved_data['ship_street'] ?? '') ?>" required>
                    </div>

                    <div class="formfield full-field"> 
                        <label>Country *</label>
                        <select id="country" name="country_code_select" required>
                            <option value="" selected>-- Select Country --</option>
                        </select>
                        <input type="hidden" id="country_hidden" name="country_code" value="<?= htmlspecialchars($displayCountryCode) ?>">
                    </div>


                    <div class="formfield full-field">
                        <label>State/Region/Province *</label>
                        <input type="text" id="state" name="state" value="<?= htmlspecialchars($step4_saved_data['state'] ?? '') ?>">
                    </div>

                    <div class="formfield step-four">
                        <div style="flex:1;">
                            <label>ZIP / Postal Code *</label>
                            <input type="text" id="zip" name="zip" value="<?= htmlspecialchars($step4_saved_data['zip'] ?? '') ?>" required>
                        </div>
                        <div style="flex:1;">
                            <label>City *</label>
                            <input type="text" id="city" name="city" value="<?= htmlspecialchars($step4_saved_data['city'] ?? '') ?>" readonly required>
                        </div>
                    </div>

                    <div class="formfield full-field" id="shipping-method">
                        <label>Shipping Method *</label>
                        <?php if (!empty($error_message) && strpos($error_message, 'Sorry, shipping options are currently unavailable') !== false): ?>
                            <p class="error-message"><?= htmlspecialchars($error_message) ?></p>
                            <input type="hidden" name="shipping_option" value="0">
                            <input type="hidden" id="shipping_price_hidden" name="shipping_price" value="0.00">
                        <?php elseif (!empty($initialShippingOptionId)): ?>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="shipping_option" id="shippingOption_<?= $initialShippingOptionId ?>" value="<?= $initialShippingOptionId ?>" checked required>
                                <label class="form-check-label" for="shippingOption_<?= $initialShippingOptionId ?>">
                                    <?= htmlspecialchars($initialShippingOptionText) ?> (£<?= number_format($initialShippingPrice, 2) ?>)
                                </label>
                            </div>
                            <input type="hidden" id="shipping_price_hidden" name="shipping_price" value="<?= number_format($initialShippingPrice, 2, '.', '') ?>">
                        <?php else: ?>
                            <p>No shipping options available for the selected country/ZIP code.</p>
                            <input type="hidden" name="shipping_option" value="0">
                            <input type="hidden" id="shipping_price_hidden" name="shipping_price" value="0.00">
                        <?php endif; ?>
                    </div>
                    
                    <p id="total-price">Total: £<?= number_format($initialTotal, 2) ?></p>

                    <div class="back-continue-button-wrap">
                        <a href="step3.php"><button type="button" class="first-btn-cicle">Back</button></a>
                        <button type="submit" class="btn secondary-btn-circle">Continue</button>
                    </div>
                </form>
            </div>

            <div class="global-container">
                <?php include('stats-card.php'); ?>
            </div>
        </div>
    </div>
</div>

<?php include('footer.php'); ?>

<script>
document.addEventListener("DOMContentLoaded", function () {
    const countrySelect = document.getElementById("country");
    const countryHidden = document.getElementById("country_hidden");
    const zipInput = document.getElementById("zip");
    const cityInput = document.getElementById("city");
    const stateInput = document.getElementById("state");
    const shippingMethodDiv = document.getElementById("shipping-method");
    const totalPriceElem = document.getElementById("total-price");
    const selectedOriginDisplay = document.getElementById("selected-origin-display");
    const shippingPriceHiddenInput = document.getElementById("shipping_price_hidden");
    
    const productPrice = <?= $productPrice ?>;
    let currentShippingPrice = parseFloat(shippingPriceHiddenInput.value || 0); 

    function escapeHtml(unsafe) {
        return unsafe
             .replace(/&/g, "&amp;")
             .replace(/</g, "&lt;")
             .replace(/>/g, "&gt;")
             .replace(/"/g, "&quot;")
             .replace(/'/g, "&#039;");
    }

    function updateTotalPriceDisplay() {
        const total = productPrice + currentShippingPrice;
        totalPriceElem.textContent = "Total: £" + total.toFixed(2);
    }

    function fetchAndDisplayShipping(countryCode, zipCode) {
        if (!countryCode) {
            shippingMethodDiv.innerHTML = "<p>Please select a country to see shipping options.</p>";
            currentShippingPrice = 0.0;
            shippingPriceHiddenInput.value = "0.00";
            updateTotalPriceDisplay();
            return;
        }

        const url = `get_shipping.php?country=${countryCode}&zip=${zipCode || ''}`;

        fetch(url)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data && data.id) {
                    shippingMethodDiv.innerHTML = `
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="shipping_option" id="shippingOption_${data.id}" value="${data.id}" checked required>
                            <label class="form-check-label" for="shippingOption_${data.id}">
                                ${escapeHtml(data.option)} (£${parseFloat(data.price).toFixed(2)})
                            </label>
                        </div>
                        <input type="hidden" name="shipping_price" value="${parseFloat(data.price)}">
                    `;
                    currentShippingPrice = parseFloat(data.price);
                } else {
                    shippingMethodDiv.innerHTML = "<p>No shipping options available for the selected country/ZIP code.</p>";
                    currentShippingPrice = 0.0;
                }
                shippingPriceHiddenInput.value = currentShippingPrice.toFixed(2); 
                updateTotalPriceDisplay();
            })
            .catch(error => {
                console.error('Error fetching shipping:', error);
                shippingMethodDiv.innerHTML = "<p class='error-message'>Error loading shipping options. Please try again later.</p>";
                currentShippingPrice = 0.0;
                shippingPriceHiddenInput.value = "0.00";
                updateTotalPriceDisplay();
            });
    }

    function fetchAndDisplayLocation(zipCode, countryCode) {
        if (!zipCode || !countryCode) {
            cityInput.value = '';
            stateInput.value = '';
            return;
        }

        fetch(`get_location.php?zip=${zipCode}&country=${countryCode}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                cityInput.value = data.city || '';
                stateInput.value = data.state || '';
            })
            .catch(error => {
                console.error('Error fetching location:', error);
                cityInput.value = '';
                stateInput.value = '';
            });
    }

    let typingTimer;
    const typingDelay = 500;

    zipInput.addEventListener("input", function () {
        clearTimeout(typingTimer);
        const zipCode = zipInput.value.trim();
        const selectedCountry = countryHidden.value;

        typingTimer = setTimeout(() => {
            fetchAndDisplayLocation(zipCode, selectedCountry);
            fetchAndDisplayShipping(selectedCountry, zipCode);
        }, typingDelay);
    });

    fetch("get_countries.php")
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(countries => {
        // Add default placeholder first
        const defaultOption = new Option("-- Select Country --", "");
        countrySelect.appendChild(defaultOption);

        countries.forEach(c => {
            if (document.getElementById("license_issue_label")) {
                const option1 = new Option(c.name, c.code);
                document.getElementById("license_issue_label").appendChild(option1);
            }
            const option2 = new Option(c.name, c.code);
            countrySelect.appendChild(option2);
        });

        // Preselect saved country if exists
        const savedCountryCode = countryHidden.value;
        if (savedCountryCode && countrySelect.querySelector(`option[value="${savedCountryCode}"]`)) {
            countrySelect.value = savedCountryCode;
        } else {
            countrySelect.value = ""; // default to placeholder
        }

        countryHidden.value = countrySelect.value;
        selectedOriginDisplay.textContent = countrySelect.options[countrySelect.selectedIndex].text || 'N/A';

        fetchAndDisplayLocation(zipInput.value.trim(), countrySelect.value);
        fetchAndDisplayShipping(countrySelect.value, zipInput.value.trim());
    })
    .catch(error => {
        console.error("Error loading countries:", error);
        if (countrySelect) countrySelect.innerHTML = '<option value="">Error loading countries</option>';
        if (selectedOriginDisplay) selectedOriginDisplay.textContent = 'Error loading';
    });

    countrySelect.addEventListener("change", () => {
        const selectedCountryCode = countrySelect.value;
        countryHidden.value = selectedCountryCode;
        selectedOriginDisplay.textContent = countrySelect.options[countrySelect.selectedIndex].text || 'N/A';

        cityInput.value = '';
        stateInput.value = '';
        fetchAndDisplayShipping(selectedCountryCode, zipInput.value.trim());
        fetchAndDisplayLocation(zipInput.value.trim(), selectedCountryCode);
    });

    updateTotalPriceDisplay();
});
</script>
</body>
</html>